import 'package:flutter/material.dart';
import 'package:get/get.dart';
import 'package:service/controllers/main_menu_controller.dart';
import 'package:service/helpers/const.dart';

class MainMenuScreen extends StatelessWidget {
  const MainMenuScreen({super.key});

  @override
  Widget build(BuildContext context) {
    final MainMenuController controller = Get.put(MainMenuController());

    return Scaffold(
      appBar: AppBar(
        title: const Text(companyName),
        actions: [
          IconButton(
            icon: const Icon(Icons.refresh),
            onPressed: () {
              controller.fetchRentalRecords();
            },
          ),
        ],
      ),
      body: SingleChildScrollView(
        padding: const EdgeInsets.all(16.0),
        child: Column(
          crossAxisAlignment: CrossAxisAlignment.start,
          children: [
            ServiceCategoryButton(
              title: 'Rental Contract Record',
              icon: Icons.receipt_long,
              count: controller.totalRental,
              color: Colors.orangeAccent,
              onTap: () async {
                await Get.toNamed('/rentalContract', arguments: {'type': '1'});
              },
              children: [
                Obx(() => InfoTile(
                      title: 'New Rentals',
                      count: controller.newRental.value,
                      onTap: () async {
                        Get.toNamed('/rentalContract',
                            arguments: {'type': '2'});
                      },
                    )),
                Obx(() => InfoTile(
                      title: 'Rentals Yet to Collect',
                      count: controller.rentalDueAlready.value,
                      onTap: () {
                        // Navigate to Rentals Yet to Collect Screen
                      },
                    )),
                Obx(() => InfoTile(
                      title: 'Rentals to Collect This Week',
                      count: controller.rentalDueThisWeek.value,
                      onTap: () {
                        // Navigate to Rentals Yet to Collect Screen
                      },
                    )),
                Obx(() => InfoTile(
                      title: 'Rental End',
                      count: controller.rentalEnd.value,
                      onTap: () {
                        // Navigate to Rental End Screen
                      },
                    )),
                Obx(() => InfoTile(
                      title: 'Rental End This Week',
                      count: controller.rentalEndThisWeek.value,
                      onTap: () {
                        // Navigate to Rental End Screen
                      },
                    )),
              ],
            ),
            const SizedBox(height: 20),

            // Maintenance Record Section
            ServiceCategoryButton(
              title: 'Maintenance Record',
              icon: Icons.build,
              count: controller.totalPms,
              color: Colors.green,
              onTap: () {
                // Navigate to Maintenance Record Screen
              },
              children: [
                Obx(() => InfoTile(
                      title: 'Due Maintenance Record',
                      count: controller.pmsDue.value,
                      onTap: () {
                        // Navigate to Due Maintenance Record Screen
                      },
                    )),
                Obx(() => InfoTile(
                      title: 'Late Maintenance Record',
                      count: controller.pmsLate.value,
                      onTap: () {
                        // Navigate to Late Maintenance Record Screen
                      },
                    )),
              ],
            ),
            const SizedBox(height: 20),

            // Summary Section
            const Text(
              'Summary',
              style: TextStyle(fontSize: 18, fontWeight: FontWeight.bold),
            ),
            const SizedBox(height: 10),
            Obx(() => SummaryCard(
                  title: 'Total Rental Contracts',
                  count: controller.totalRental.value,
                  color: Colors.orangeAccent,
                )),
            Obx(() => SummaryCard(
                  title: 'Total Maintenance Records',
                  count: controller.totalPms.value,
                  color: Colors.green,
                )),
          ],
        ),
      ),
    );
  }
}

class ServiceCategoryButton extends StatelessWidget {
  final String title;
  final IconData icon;
  final RxInt count;
  final Color color;
  final VoidCallback onTap;
  final List<Widget> children;

  const ServiceCategoryButton({
    super.key,
    required this.title,
    required this.icon,
    required this.count,
    required this.color,
    required this.onTap,
    required this.children,
  });

  @override
  Widget build(BuildContext context) {
    return GestureDetector(
      onTap: onTap,
      child: Container(
        padding: const EdgeInsets.all(16.0),
        decoration: BoxDecoration(
          color: color.withOpacity(0.1),
          borderRadius: BorderRadius.circular(12),
          border: Border.all(color: color, width: 2),
        ),
        child: Column(
          children: [
            Row(
              children: [
                Icon(icon, size: 40, color: color),
                const SizedBox(width: 16),
                Expanded(
                  child: Text(
                    title,
                    style: const TextStyle(
                        fontSize: 18, fontWeight: FontWeight.bold),
                  ),
                ),
                Obx(() => CircleAvatar(
                      backgroundColor: color,
                      child: Text(
                        count.value.toString(),
                        style: const TextStyle(color: Colors.white),
                      ),
                    )),
              ],
            ),
            const SizedBox(height: 10),
            Column(children: children),
          ],
        ),
      ),
    );
  }
}

class InfoTile extends StatelessWidget {
  final String title;
  final int count;
  final VoidCallback onTap;

  const InfoTile({
    super.key,
    required this.title,
    required this.count,
    required this.onTap,
  });

  @override
  Widget build(BuildContext context) {
    return GestureDetector(
      onTap: onTap,
      child: ListTile(
        title: Text(title),
        trailing: CircleAvatar(
          backgroundColor: Colors.blueAccent,
          child: Text(
            count.toString(),
            style: const TextStyle(color: Colors.white),
          ),
        ),
      ),
    );
  }
}

class SummaryCard extends StatelessWidget {
  final String title;
  final int count;
  final Color color;

  const SummaryCard({
    super.key,
    required this.title,
    required this.count,
    required this.color,
  });

  @override
  Widget build(BuildContext context) {
    return Card(
      margin: const EdgeInsets.symmetric(vertical: 10),
      shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(10)),
      elevation: 5,
      child: ListTile(
        leading: Icon(Icons.analytics, size: 40, color: color),
        title: Text(
          title,
          style: const TextStyle(fontSize: 18, fontWeight: FontWeight.bold),
        ),
        trailing: CircleAvatar(
          backgroundColor: color,
          child: Text(
            count.toString(),
            style: const TextStyle(color: Colors.white),
          ),
        ),
      ),
    );
  }
}
